<?php
/**
 * User Dashboard Page (Hub for Past Reports and New Interviews)
 *
 * This is the main hub for logged-in users. It displays their past interview
 * reports and provides a clear call-to-action to start a new interview session.
 */

include THEME_PATH . 'header.php';

// --- DATA FETCHING FOR USER STATE & HISTORY ---
$current_step = 1;
$past_reports = [];
$error_message = '';

if (is_logged_in()) {
    try {
        $pdo = get_db_connection();
        $user_id = $_SESSION['user_id'];

        // Get the user's current step
        $stmt_step = $pdo->prepare("SELECT current_step FROM users WHERE id = ?");
        $stmt_step->execute([$user_id]);
        $current_step = $stmt_step->fetchColumn();

        // Get all past interview reports for this user
        $stmt_reports = $pdo->prepare("SELECT id, generated_at FROM interview_reports WHERE user_id = ? ORDER BY generated_at DESC");
        $stmt_reports->execute([$user_id]);
        $past_reports = $stmt_reports->fetchAll();

    } catch (PDOException $e) {
        $error_message = "Could not load your dashboard. Please try again later.";
    }
}
?>

<div class="user-dashboard">
    <div class="container">

        <!-- Header Section -->
        <div class="dashboard-header">
            <div>
                <h1 class="greeting"><?php echo trans('dashboard_greeting'); ?> <?php echo e($_SESSION['user_name'] ?? 'User'); ?>!</h1>
                <p class="subtitle"><?php echo trans('dashboard_welcome_back'); ?></p>
            </div>
            <!-- Language Selector -->
            <form action="index.php?page=dashboard" method="POST" class="language-selector">
                <label for="lang-select"><?php echo trans('language_selector_label'); ?></label>
                <select name="lang" id="lang-select" onchange="this.form.submit()">
                    <?php
                    $current_lang = $_SESSION['lang'] ?? DEFAULT_LANG;
                    foreach (SUPPORTED_LANGUAGES as $code => $name) {
                        $selected = ($code === $current_lang) ? 'selected' : '';
                        echo "<option value=\"{$code}\" {$selected}>{$name}</option>";
                    }
                    ?>
                </select>
            </form>
        </div>

        <?php if ($error_message): ?>
            <div class="alert alert-danger"><?php echo e($error_message); ?></div>
        <?php endif; ?>

        <!-- New Interview Section -->
        <div class="new-interview-section">
            <h2><?php echo trans('dashboard_ready_for_next'); ?></h2>
            <p><?php echo trans('dashboard_ready_subtitle'); ?></p>
            <a href="index.php?page=upload" class="btn btn-primary btn-lg"><i class="fas fa-play-circle"></i> <?php echo trans('dashboard_start_new'); ?></a>
        </div>

        <!-- Interview History Section -->
        <div class="history-section">
            <h2><i class="fas fa-history"></i> <?php echo trans('dashboard_history_title'); ?></h2>
            <?php if (empty($past_reports)): ?>
                <div class="empty-state">
                    <p><?php echo trans('dashboard_no_history'); ?></p>
                </div>
            <?php else: ?>
                <div class="report-list">
                    <?php foreach ($past_reports as $report): ?>
                        <div class="report-item">
                            <div class="report-icon"><i class="fas fa-file-alt"></i></div>
                            <div class="report-details">
                                <strong><?php echo trans('dashboard_report_title'); ?></strong>
                                <span><?php echo trans('dashboard_report_completed'); ?>: <?php echo date('d F Y, H:i', strtotime($report['generated_at'])); ?></span>
                            </div>
                            <a href="index.php?page=results&id=<?php echo $report['id']; ?>" class="btn btn-secondary"><?php echo trans('dashboard_view_report'); ?></a>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>

    </div>
</div>
<style>
.new-interview-section {
    background: var(--card-bg);
    padding: 40px;
    border-radius: 12px;
    text-align: center;
    margin-bottom: 40px;
    box-shadow: 0 8px 25px rgba(74, 144, 226, 0.1);
}
.new-interview-section h2 { margin-top: 0; font-size: 28px; }
.btn-lg { padding: 15px 30px; font-size: 18px; }

.history-section h2 { font-size: 24px; margin-bottom: 20px; }
.report-list { display: flex; flex-direction: column; gap: 15px; }
.report-item {
    display: flex;
    align-items: center;
    background: var(--card-bg);
    padding: 20px;
    border-radius: 12px;
    border: 1px solid var(--border-color);
    transition: all 0.2s ease;
}
.report-item:hover { border-color: var(--primary-color); transform: translateY(-3px); box-shadow: 0 4px 15px rgba(0,0,0,0.07); }
.report-icon { font-size: 28px; color: var(--primary-color); margin-right: 20px; }
.report-details { flex-grow: 1; }
.report-details strong { display: block; font-size: 18px; }
.report-details span { font-size: 14px; color: var(--text-secondary); }
.empty-state {
    text-align: center;
    padding: 40px;
    background: #f8f9fa;
    border: 2px dashed var(--border-color);
    border-radius: 12px;
}
</style>
<?php
include THEME_PATH . 'footer.php';
?>

