<?php
/**
 * Admin User Management Page
 *
 * This page allows administrators to view, search, and manage all
 * registered users on the platform.
 */

// The admin router (admin/index.php) handles the session and security checks.
include THEME_PATH . 'admin/header.php'; // Include the admin header

// --- DATA FETCHING FOR USERS ---
$users = [];
try {
    $pdo = get_db_connection();

    // Fetch all users from the database, ordering by the most recent.
    $stmt = $pdo->query("
        SELECT id, full_name, email, email_verified, created_at
        FROM users
        ORDER BY created_at DESC
    ");
    $users = $stmt->fetchAll();

} catch (PDOException $e) {
    $db_error = "Could not fetch user data. Error: " . $e->getMessage();
}

?>

<div class="admin-content">
    <div class="container">
        <div class="page-header">
            <h1 class="page-title">User Management</h1>
            <!-- FIXED: The href now points to the correct page for adding a new user -->
            <a href="index.php?page=add_user" class="btn btn-success"><i class="fas fa-user-plus"></i> Add New User</a>
        </div>
        <p class="page-subtitle">View, edit, and manage all registered user accounts.</p>

        <!-- Session Message Display (for add/edit/delete actions) -->
        <?php if (isset($_SESSION['success_message'])): ?>
            <div class="alert alert-success"><?php echo e($_SESSION['success_message']); unset($_SESSION['success_message']); ?></div>
        <?php endif; ?>
        <?php if (isset($_SESSION['error_message'])): ?>
            <div class="alert alert-danger"><?php echo e($_SESSION['error_message']); unset($_SESSION['error_message']); ?></div>
        <?php endif; ?>
        <?php if (isset($db_error)): ?>
            <div class="alert alert-danger"><?php echo e($db_error); ?></div>
        <?php endif; ?>

        <!-- Users Table -->
        <div class="data-table-container">
            <table class="data-table">
                <thead>
                    <tr>
                        <th>Full Name</th>
                        <th>Email Address</th>
                        <th>Status</th>
                        <th>Registration Date</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($users)): ?>
                        <tr>
                            <td colspan="5" class="text-center">No users found.</td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($users as $user): ?>
                            <tr>
                                <td data-label="Full Name"><strong><?php echo e($user['full_name']); ?></strong></td>
                                <td data-label="Email Address"><?php echo e($user['email']); ?></td>
                                <td data-label="Status">
                                    <?php if ($user['email_verified']): ?>
                                        <span class="badge badge-success"><i class="fas fa-check-circle"></i> Verified</span>
                                    <?php else: ?>
                                        <span class="badge badge-warning"><i class="fas fa-exclamation-triangle"></i> Pending</span>
                                    <?php endif; ?>
                                </td>
                                <td data-label="Registration Date">
                                    <?php
                                        $date = new DateTime($user['created_at']);
                                        echo $date->format('d M Y, H:i');
                                    ?>
                                </td>
                                <td data-label="Actions">
                                    <!-- FIXED: The href now points to the correct page for editing a user -->
                                    <a href="index.php?page=edit_user&id=<?php echo e($user['id']); ?>" class="btn-action btn-edit"><i class="fas fa-edit"></i> Edit</a>
                                    
                                    <!-- Delete button within a form for security -->
                                    <form action="../actions/user_management_handler.php" method="POST" style="display: inline-block;">
                                        <input type="hidden" name="action" value="delete_user">
                                        <input type="hidden" name="user_id" value="<?php echo e($user['id']); ?>">
                                        <button type="submit" class="btn-action btn-delete" onclick="return confirm('Are you sure you want to delete this user? This action cannot be undone.');">
                                            <i class="fas fa-trash-alt"></i> Delete
                                        </button>
                                    </form>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>

    </div>
</div>

<?php
// Include the admin footer template.
include THEME_PATH . 'admin/footer.php';
?>

