<?php
/**
 * Admin Sessions / Activity Log Page
 *
 * This page displays a chronological log of important events that have
 * occurred on the platform, such as new user registrations and AI learning updates.
 */

// The admin router (admin/index.php) handles the session and security checks.
include THEME_PATH . 'admin/header.php'; // Include the admin header

// --- DATA FETCHING FOR ACTIVITY LOG ---
$activities = [];
try {
    $pdo = get_db_connection();

    // 1. Fetch the 20 most recent user registrations.
    $stmt_users = $pdo->query("
        SELECT full_name, email, created_at AS timestamp, 'New User Registration' AS type
        FROM users
        ORDER BY created_at DESC
        LIMIT 20
    ");
    $user_activities = $stmt_users->fetchAll();

    // 2. Fetch the 20 most recent AI learning activities.
    $stmt_learning = $pdo->query("
        SELECT file_name, processed_at AS timestamp, 'AI Learning Update' AS type
        FROM processed_files
        ORDER BY processed_at DESC
        LIMIT 20
    ");
    $learning_activities = $stmt_learning->fetchAll();

    // 3. Merge the two activity lists into one.
    $activities = array_merge($user_activities, $learning_activities);

    // 4. Sort the merged list chronologically by timestamp.
    usort($activities, function($a, $b) {
        return strtotime($b['timestamp']) - strtotime($a['timestamp']);
    });

    // 5. Limit the final list to the 30 most recent activities overall.
    $activities = array_slice($activities, 0, 30);

} catch (PDOException $e) {
    $db_error = "Could not fetch activity log. Error: " . $e->getMessage();
}

?>

<div class="admin-content">
    <div class="container">
        <h1 class="page-title">Activity Log</h1>
        <p class="page-subtitle">A chronological overview of recent events on the platform.</p>

        <?php if (isset($db_error)): ?>
            <div class="alert alert-danger"><?php echo e($db_error); ?></div>
        <?php endif; ?>

        <!-- Activity Log Table -->
        <div class="data-table-container">
            <table class="data-table">
                <thead>
                    <tr>
                        <th>Timestamp</th>
                        <th>Activity Type</th>
                        <th>Details</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($activities)): ?>
                        <tr>
                            <td colspan="3" class="text-center">No recent activity found.</td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($activities as $activity): ?>
                            <tr>
                                <td data-label="Timestamp">
                                    <?php
                                        // Format the timestamp for better readability
                                        $date = new DateTime($activity['timestamp'], new DateTimeZone('UTC'));
                                        $date->setTimezone(new DateTimeZone('Asia/Jakarta')); // Convert to WIB
                                        echo $date->format('d M Y, H:i:s');
                                    ?>
                                </td>
                                <td data-label="Activity Type">
                                    <?php if ($activity['type'] === 'New User Registration'): ?>
                                        <span class="badge badge-success"><i class="fas fa-user-plus"></i> <?php echo e($activity['type']); ?></span>
                                    <?php elseif ($activity['type'] === 'AI Learning Update'): ?>
                                        <span class="badge badge-info"><i class="fas fa-brain"></i> <?php echo e($activity['type']); ?></span>
                                    <?php else: ?>
                                        <span class="badge badge-secondary"><?php echo e($activity['type']); ?></span>
                                    <?php endif; ?>
                                </td>
                                <td data-label="Details">
                                    <?php
                                        // Display different details based on the activity type
                                        if ($activity['type'] === 'New User Registration') {
                                            echo 'User: <strong>' . e($activity['full_name']) . '</strong> (' . e($activity['email']) . ')';
                                        } elseif ($activity['type'] === 'AI Learning Update') {
                                            echo 'Learned from file: <strong>' . e($activity['file_name']) . '</strong>';
                                        }
                                    ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>

    </div>
</div>

<?php
// Include the admin footer template.
include THEME_PATH . 'admin/footer.php';
?>
