<?php
/**
 * Admin Manage Questions Page
 *
 * This page allows an administrator to add, view, and delete questions
 * and their corresponding answers for a specific psychological test.
 */

// The admin router (admin/index.php) handles the session and security checks.
include THEME_PATH . 'admin/header.php';

// 1. Get Test ID from URL and Fetch Data
// -----------------------------------------------------------------------------
$test_id = filter_input(INPUT_GET, 'test_id', FILTER_VALIDATE_INT);
$test = null;
$questions = [];
$error_message = '';

if ($test_id) {
    try {
        $pdo = get_db_connection();
        
        // Fetch the main test details
        $stmt_test = $pdo->prepare("SELECT id, test_name FROM psychological_tests WHERE id = ?");
        $stmt_test->execute([$test_id]);
        $test = $stmt_test->fetch();

        if (!$test) {
            $error_message = "Test not found. It may have been deleted.";
        } else {
            // Fetch all questions and their associated answers for this test
            $stmt_questions = $pdo->prepare("
                SELECT q.id AS question_id, q.question_text, a.id AS answer_id, a.answer_text, a.score
                FROM test_questions q
                LEFT JOIN test_answers a ON q.id = a.question_id
                WHERE q.test_id = ?
                ORDER BY q.id, a.id
            ");
            $stmt_questions->execute([$test_id]);
            $results = $stmt_questions->fetchAll();

            // Group answers by question
            foreach ($results as $row) {
                if (!isset($questions[$row['question_id']])) {
                    $questions[$row['question_id']] = [
                        'question_text' => $row['question_text'],
                        'answers' => []
                    ];
                }
                if ($row['answer_id']) {
                    $questions[$row['question_id']]['answers'][] = [
                        'answer_text' => $row['answer_text'],
                        'score' => $row['score']
                    ];
                }
            }
        }
    } catch (PDOException $e) {
        $error_message = "A database error occurred: " . $e->getMessage();
    }
} else {
    $error_message = "Invalid or missing Test ID provided in the URL.";
}
?>

<div class="admin-content">
    <div class="container">
        
        <?php if ($error_message): ?>
            <h1 class="page-title">Error</h1>
            <p class="page-subtitle">Could not load question data.</p>
            <div class="alert alert-danger"><?php echo e($error_message); ?></div>
            <a href="index.php?page=tests" class="btn btn-secondary"><i class="fas fa-arrow-left"></i> Back to Test List</a>
        <?php elseif ($test): ?>
            <div class="page-header">
                <div>
                    <h1 class="page-title">Manage Questions</h1>
                    <p class="page-subtitle">For test: <strong><?php echo e($test['test_name']); ?></strong></p>
                </div>
                <a href="index.php?page=tests" class="btn btn-secondary"><i class="fas fa-arrow-left"></i> Back to Test List</a>
            </div>

            <!-- Session Message Display -->
            <?php if (isset($_SESSION['success_message'])): ?>
                <div class="alert alert-success"><?php echo e($_SESSION['success_message']); unset($_SESSION['success_message']); ?></div>
            <?php endif; ?>
            <?php if (isset($_SESSION['error_message'])): ?>
                <div class="alert alert-danger"><?php echo e($_SESSION['error_message']); unset($_SESSION['error_message']); ?></div>
            <?php endif; ?>

            <!-- Add New Question Form -->
            <div class="form-container" style="margin-bottom: 40px;">
                <h2 class="section-title">Add New Question</h2>
                <form action="../actions/test_management_handler.php" method="POST">
                    <input type="hidden" name="action" value="add_question">
                    <input type="hidden" name="test_id" value="<?php echo e($test['id']); ?>">
                    
                    <div class="form-group">
                        <label for="question_text">Question Text</label>
                        <textarea id="question_text" name="question_text" class="form-control" rows="3" placeholder="Enter the full question here..." required></textarea>
                    </div>

                    <div id="answers-container">
                        <div class="answer-group">
                            <div class="form-group">
                                <label>Answer 1</label>
                                <input type="text" name="answers[]" class="form-control" placeholder="Answer text" required>
                            </div>
                            <div class="form-group-score">
                                <label>Score</label>
                                <input type="number" name="scores[]" class="form-control" placeholder="e.g., 5" value="0" required>
                            </div>
                        </div>
                        <div class="answer-group">
                             <div class="form-group">
                                <label>Answer 2</label>
                                <input type="text" name="answers[]" class="form-control" placeholder="Answer text" required>
                            </div>
                            <div class="form-group-score">
                                <label>Score</label>
                                <input type="number" name="scores[]" class="form-control" placeholder="e.g., 3" value="0" required>
                            </div>
                        </div>
                    </div>

                    <div class="form-actions">
                        <button type="button" id="add-answer-btn" class="btn btn-secondary"><i class="fas fa-plus"></i> Add Another Answer</button>
                        <button type="submit" class="btn btn-success"><i class="fas fa-save"></i> Save Question</button>
                    </div>
                </form>
            </div>

            <!-- Existing Questions List -->
            <h2 class="section-title">Existing Questions (<?php echo count($questions); ?>)</h2>
            <div class="questions-list">
                <?php if (empty($questions)): ?>
                    <p>No questions have been added to this test yet.</p>
                <?php else: ?>
                    <?php foreach ($questions as $question_id => $q_data): ?>
                        <div class="question-card">
                            <div class="question-header">
                                <p class="question-text"><?php echo e($q_data['question_text']); ?></p>
                                <form action="../actions/test_management_handler.php" method="POST" onsubmit="return confirm('Are you sure you want to delete this question and all its answers?');">
                                    <input type="hidden" name="action" value="delete_question">
                                    <input type="hidden" name="question_id" value="<?php echo e($question_id); ?>">
                                    <input type="hidden" name="test_id" value="<?php echo e($test['id']); ?>">
                                    <button type="submit" class="btn-action btn-delete"><i class="fas fa-trash-alt"></i> Delete</button>
                                </form>
                            </div>
                            <ul class="answers-list">
                                <?php foreach ($q_data['answers'] as $answer): ?>
                                    <li>
                                        <span><?php echo e($answer['answer_text']); ?></span>
                                        <span class="badge badge-info">Score: <?php echo e($answer['score']); ?></span>
                                    </li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<style>
    /* Custom styles for this page */
    .answer-group { display: flex; gap: 15px; align-items: flex-end; }
    .answer-group .form-group { flex-grow: 1; }
    .answer-group .form-group-score { width: 100px; flex-shrink: 0; }
    .questions-list { display: flex; flex-direction: column; gap: 20px; }
    .question-card { background: var(--card-bg); border-radius: 8px; border: 1px solid var(--border-color); }
    .question-header { display: flex; justify-content: space-between; align-items: center; padding: 15px; border-bottom: 1px solid var(--border-color); }
    .question-text { margin: 0; font-weight: 600; }
    .answers-list { list-style: none; padding: 0; margin: 0; }
    .answers-list li { display: flex; justify-content: space-between; padding: 10px 15px; border-bottom: 1px solid #f0f0f0; }
    .answers-list li:last-child { border-bottom: none; }
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const addAnswerBtn = document.getElementById('add-answer-btn');
    const answersContainer = document.getElementById('answers-container');
    let answerCount = 2;

    addAnswerBtn.addEventListener('click', function() {
        answerCount++;
        const newAnswerGroup = document.createElement('div');
        newAnswerGroup.className = 'answer-group';
        newAnswerGroup.innerHTML = `
            <div class="form-group">
                <label>Answer ${answerCount}</label>
                <input type="text" name="answers[]" class="form-control" placeholder="Answer text" required>
            </div>
            <div class="form-group-score">
                <label>Score</label>
                <input type="number" name="scores[]" class="form-control" value="0" required>
            </div>
        `;
        answersContainer.appendChild(newAnswerGroup);
    });
});
</script>

<?php
include THEME_PATH . 'admin/footer.php';
?>

