<?php
/**
 * AI Learning Management Page (Updated with Real-time Job Queue Monitoring)
 *
 * This page provides the UI for an administrator to trigger the AI learning process.
 * It now uses AJAX to poll for the status of background jobs and displays
 * real-time progress to the user.
 */

// The admin router (admin/index.php) handles the session and security checks.
include THEME_PATH . 'admin/header.php'; // Include the admin header

?>

<!-- Add custom styles for the real-time status display -->
<style>
    .status-container {
        margin-top: 25px;
        padding: 20px;
        border: 1px solid var(--border-color);
        border-radius: 8px;
        background-color: #f8f9fa;
    }
    .status-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 15px;
    }
    .status-label {
        font-weight: 600;
        color: var(--text-secondary);
    }
    .status-text {
        font-weight: 700;
        padding: 5px 12px;
        border-radius: 20px;
        font-size: 12px;
        text-transform: capitalize;
    }
    .status-pending { background-color: #fff8e1; color: #f57c00; }
    .status-processing { background-color: #e7f3fe; color: var(--primary-color); }
    .status-completed { background-color: #eaf7ed; color: var(--success-color); }
    .status-failed { background-color: #f8d7da; color: var(--danger-color); }
    .status-not_found { background-color: #f0f0f0; color: #888; }

    .progress-bar-container {
        height: 8px;
        width: 100%;
        background-color: #e9ecef;
        border-radius: 4px;
        overflow: hidden;
        margin-bottom: 10px;
    }
    .progress-bar-dynamic {
        height: 100%;
        width: 0%;
        background-color: var(--primary-color);
        border-radius: 4px;
        transition: width 0.5s ease;
    }
    .status-message {
        font-size: 13px;
        color: var(--text-secondary);
        margin: 0;
        min-height: 20px;
    }
    .btn-learn:disabled {
        opacity: 0.6;
        cursor: not-allowed;
    }
</style>

<div class="admin-content">
    <div class="container">
        <h1 class="page-title">Update AI Learnings</h1>
        <p class="page-subtitle">
            Initiate the AI learning process. The task will run in the background, and you can monitor its progress below.
        </p>

        <!-- Session Message Display -->
        <?php if (isset($_SESSION['success_message'])): ?>
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i>
                <?php echo e($_SESSION['success_message']); unset($_SESSION['success_message']); ?>
            </div>
        <?php elseif (isset($_SESSION['info_message'])): ?>
             <div class="alert alert-info" style="background-color: #e7f3fe; color: #0c5460; border: 1px solid #bde5f8;">
                <i class="fas fa-info-circle"></i>
                <?php echo e($_SESSION['info_message']); unset($_SESSION['info_message']); ?>
            </div>
        <?php elseif (isset($_SESSION['error_message'])): ?>
            <div class="alert alert-danger">
                <i class="fas fa-exclamation-triangle"></i>
                <?php echo e($_SESSION['error_message']); unset($_SESSION['error_message']); ?>
            </div>
        <?php endif; ?>

        <!-- Learning Actions Grid -->
        <div class="learning-actions-grid">
            <!-- Card for Job Posters -->
            <div class="action-card" id="job_poster-card">
                <div class="action-card-icon"><i class="fas fa-briefcase"></i></div>
                <h2 class="action-card-title">Job Posters</h2>
                <p class="action-card-description">Scan the 'Job Posters' folder for new files to learn from.</p>
                <form action="../actions/update_learnings_handler.php" method="POST">
                    <input type="hidden" name="folder_type" value="job_poster">
                    <button type="submit" class="btn btn-primary btn-learn">
                        <i class="fas fa-sync-alt"></i> Update Learnings
                    </button>
                </form>
                <!-- Real-time status display will be injected here -->
                <div class="status-container" id="job_poster-status-container" style="display: none;">
                    <div class="status-header">
                        <span class="status-label">Last Status:</span>
                        <span class="status-text" id="job_poster-status-text"></span>
                    </div>
                    <div class="progress-bar-container">
                        <div class="progress-bar-dynamic" id="job_poster-progress-bar"></div>
                    </div>
                    <p class="status-message" id="job_poster-message"></p>
                </div>
            </div>

            <!-- Card for CVs -->
            <div class="action-card" id="cv-card">
                <div class="action-card-icon"><i class="fas fa-file-invoice"></i></div>
                <h2 class="action-card-title">Candidate CVs</h2>
                <p class="action-card-description">Scan the 'CVs' folder for new candidate resumes to learn from.</p>
                <form action="../actions/update_learnings_handler.php" method="POST">
                    <input type="hidden" name="folder_type" value="cv">
                    <button type="submit" class="btn btn-primary btn-learn">
                        <i class="fas fa-sync-alt"></i> Update Learnings
                    </button>
                </form>
                <!-- Real-time status display will be injected here -->
                 <div class="status-container" id="cv-status-container" style="display: none;">
                    <div class="status-header">
                        <span class="status-label">Last Status:</span>
                        <span class="status-text" id="cv-status-text"></span>
                    </div>
                    <div class="progress-bar-container">
                        <div class="progress-bar-dynamic" id="cv-progress-bar"></div>
                    </div>
                    <p class="status-message" id="cv-message"></p>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    /**
     * AI Learning Real-time Status Updater
     * This function polls the server to get the latest status of background jobs.
     */
    function updateStatus(jobType) {
        fetch(`../actions/get_learning_status.php?type=${jobType}`)
            .then(response => {
                if (!response.ok) {
                    throw new Error('Network response was not ok');
                }
                return response.json();
            })
            .then(data => {
                if (data.status !== 'success') {
                    console.error('API Error:', data.message);
                    return;
                }
                
                // Get elements for the specific card
                const container = document.getElementById(`${jobType}-status-container`);
                const statusText = document.getElementById(`${jobType}-status-text`);
                const progressBar = document.getElementById(`${jobType}-progress-bar`);
                const messageText = document.getElementById(`${jobType}-message`);
                const button = document.querySelector(`#${jobType}-card .btn-learn`);

                // Show the container if a job has ever been run
                if (data.job_status && data.job_status !== 'not_found') {
                    container.style.display = 'block';
                } else {
                    container.style.display = 'none';
                    return; // No job to display status for
                }

                // Update status text and apply class for color
                statusText.textContent = data.job_status.replace('_', ' ');
                statusText.className = `status-text status-${data.job_status}`;
                
                // Update progress bar width
                progressBar.style.width = `${data.progress_percent || 0}%`;
                
                // Update informative message
                let message = data.message || '';
                if (data.job_status === 'processing') {
                    message = `Processing... Scanned ${data.processed_items} of ${data.total_items} items.`;
                }
                messageText.textContent = message;

                // Enable or disable the button based on job status
                if (data.job_status === 'pending' || data.job_status === 'processing') {
                    button.disabled = true;
                    button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> In Progress...';
                } else {
                    button.disabled = false;
                    button.innerHTML = '<i class="fas fa-sync-alt"></i> Update Learnings';
                }
            })
            .catch(error => {
                console.error(`Error fetching status for ${jobType}:`, error);
                // Optionally, update the UI to show an error state
                const container = document.getElementById(`${jobType}-status-container`);
                if (container) {
                    container.style.display = 'block';
                    const messageText = document.getElementById(`${jobType}-message`);
                    if(messageText) messageText.textContent = 'Could not fetch status. Check console.';
                }
            });
    }

    // Immediately check status on page load
    updateStatus('job_poster');
    updateStatus('cv');

    // Poll for new status every 5 seconds
    setInterval(() => {
        updateStatus('job_poster');
        updateStatus('cv');
    }, 5000);
});
</script>

<?php
// Include the admin footer template.
include THEME_PATH . 'admin/footer.php';
?>

