<?php
/**
 * User Page: Psychological Test (Step 3)
 *
 * Ini adalah file "container" utama untuk Step 3.
 *
 * MODIFIKASI: File ini sekarang cerdas. Ia akan:
 * 1. Mencari tahu tes apa yang harus diambil pengguna (logika kustom Anda bisa masuk di sini).
 * 2. Memeriksa 'test_type' (simple, disc, papi) dari tes tersebut.
 * 3. Memuat file partial yang benar (simple, disc, atau papi) ke dalam halaman.
 * 4. Mengarahkan form ke handler baru yang bisa memproses semua tipe tes.
 */

// 1. Inisialisasi & Keamanan
// (Kita asumsikan main_header.php atau index.php sudah menjalankan session_start())

if (!is_logged_in()) {
    header('Location: index.php?page=login');
    exit();
}

// Pastikan pengguna berada di langkah yang benar
// (upload_handler.php seharusnya mengatur current_step = 3)
if ($_SESSION['current_step'] < 3) { 
    // Jika mereka mencoba melompat, kembalikan ke dashboard
    header('Location: index.php?page=dashboard');
    exit();
}

$user_id = $_SESSION['user_id'];
$test_to_take = null;
$test_type = 'simple'; // Default
$test_id = null;
$test_name = 'Psychological Test';

// Ambil pesan error jika ada (dari submit sebelumnya yang gagal)
$error_message = $_SESSION['error_message'] ?? null;
unset($_SESSION['error_message']); // Hapus setelah ditampilkan

// 2. Logika untuk Menentukan Tes yang Akan Diambil
try {
    $pdo = get_db_connection();
    
    // --- Logika Penentuan Tes ---
    // Logika ini harus disesuaikan dengan alur bisnis Anda.
    // Apakah admin menugaskan tes? Apakah semua orang mengambil tes yang sama?
    
    // UNTUK CONTOH INI: Kita akan mencari tes 'DISC' terlebih dahulu.
    // Jika tidak ada, baru kita ambil tes 'simple' pertama yang ada.
    
    // Coba cari tes DISC yang aktif
    $stmt = $pdo->prepare("SELECT id, test_name, test_type FROM psychological_tests WHERE test_type = 'disc' LIMIT 1");
    $stmt->execute();
    $test_to_take = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$test_to_take) {
        // Jika tidak ada tes DISC, coba cari tes PAPI
        $stmt = $pdo->prepare("SELECT id, test_name, test_type FROM psychological_tests WHERE test_type = 'papi' LIMIT 1");
        $stmt->execute();
        $test_to_take = $stmt->fetch(PDO::FETCH_ASSOC);
    }

    if (!$test_to_take) {
        // Jika keduanya tidak ada, ambil tes 'simple' pertama
        $stmt = $pdo->prepare("SELECT id, test_name, test_type FROM psychological_tests WHERE test_type = 'simple' LIMIT 1");
        $stmt->execute();
        $test_to_take = $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    // Jika tes ditemukan, set variabel
    if ($test_to_take) {
        $test_id = $test_to_take['id'];
        $test_name = $test_to_take['test_name'];
        $test_type = $test_to_take['test_type'];
    } else {
        // Tidak ada tes sama sekali di database
        throw new Exception("No psychological tests are configured in the system. Please contact admin.");
    }

} catch (Exception $e) {
    $error_message = "Error loading test: " . $e->getMessage();
    $test_to_take = null; // Mencegah sisa halaman render
}

// 3. Render Halaman
// (main_header.php akan di-include oleh index.php)
?>

<div class="container mt-5 mb-5">
    <div class="row justify-content-center">
        <div class="col-lg-10">
            
            <div class="card shadow-sm">
                <div class="card-header bg-light">
                    <h3><i class="fas fa-brain"></i> Step 3: Psychological Test</h3>
                </div>
                <div class="card-body p-4 p-md-5">

                    <?php if ($error_message): ?>
                        <div class="alert alert-danger">
                            <?php echo htmlspecialchars($error_message); ?>
                        </div>
                    <?php endif; ?>

                    <?php if ($test_to_take): ?>
                        
                        <h2 class="text-center mb-3"><?php echo htmlspecialchars($test_name); ?></h2>
                        
                        <form id="psyTestForm" action="actions/submit_psychotest_handler.php" method="POST">
                            
                            <input type="hidden" name="test_id" value="<?php echo (int)$test_id; ?>">

                            <?php
                            // =======================================================
                            // LOGIKA KONDISIONAL UNTUK MEMUAT PARTIAL YANG TEPAT
                            // =======================================================
                            
                            if ($test_type === 'disc') {
                                
                                // Muat formulir DISC
                                include 'partials/psy_test_disc.php';
                                
                            } elseif ($test_type === 'papi') {
                                
                                // Muat formulir PAPI
                                include 'partials/psy_test_papi.php';
                                
                            } else {
                                
                                // --- INI ADALAH LOGIKA UNTUK TIPE 'SIMPLE' (LOGIKA LAMA) ---
                                // Ini menggantikan file `partials/psy_test_simple.php`
                                
                                // Ambil semua pertanyaan dan jawaban untuk tes 'simple' ini
                                $stmt_q = $pdo->prepare("SELECT id, question_text FROM test_questions WHERE test_id = ? ORDER BY id ASC");
                                $stmt_q->execute([$test_id]);
                                $questions = $stmt_q->fetchAll(PDO::FETCH_ASSOC);
                                
                                $stmt_a = $pdo->prepare("SELECT id, answer_text FROM test_answers WHERE question_id = ?");
                                
                                if (empty($questions)) {
                                    echo "<div class='alert alert-warning'>This test ('simple') has no questions configured.</div>";
                                } else {
                                    echo '<div class="test-instructions mb-4">';
                                    echo '<h4>Petunjuk Pengerjaan</h4>';
                                    echo '<p>Pilih satu jawaban yang paling sesuai untuk setiap pertanyaan di bawah ini.</p>';
                                    echo '</div><hr>';
                                    
                                    $q_index = 1;
                                    foreach ($questions as $question) {
                                        echo '<fieldset class="form-group mb-4">';
                                        echo '<legend class="h5" style="font-size: 1.1em; font-weight: bold;">' . $q_index . '. ' . htmlspecialchars($question['question_text']) . '</legend>';
                                        
                                        // Ambil jawaban untuk pertanyaan ini
                                        $stmt_a->execute([$question['id']]);
                                        $answers = $stmt_a->fetchAll(PDO::FETCH_ASSOC);
                                        
                                        if (empty($answers)) {
                                            echo '<p class="text-danger">Error: No answers found for this question.</p>';
                                        } else {
                                            foreach ($answers as $answer) {
                                                // Nama 'answers[question_id]' = 'answer_id'
                                                // Ini adalah format yang diharapkan oleh handler 'simple' 
                                                $radio_id = 'ans_' . $answer['id'];
                                                echo '<div class="form-check">';
                                                echo '<input class="form-check-input" type="radio" name="answers[' . $question['id'] . ']" id="' . $radio_id . '" value="' . $answer['id'] . '" required>';
                                                echo '<label class="form-check-label" for="' . $radio_id . '">';
                                                echo htmlspecialchars($answer['answer_text']);
                                                echo '</label>';
                                                echo '</div>';
                                            }
                                        }
                                        echo '</fieldset>';
                                        $q_index++;
                                    }
                                }
                            }
                            ?>

                            <hr class="mt-5">
                            <div class="text-right">
                                <button type="submit" class="btn btn-primary btn-lg">
                                    Selesai & Lanjutkan <i class="fas fa-arrow-right ml-2"></i>
                                </button>
                            </div>
                            
                        </form>

                    <?php else: ?>
                        <div class="alert alert-danger">
                            Sistem tidak dapat memuat tes psikologi. Silakan hubungi administrator.
                        </div>
                    <?php endif; ?>

                </div> </div> </div> </div> </div> <?php
// (main_footer.php akan di-include oleh index.php)
?>