<?php
/**
 * AI Learning Cron Job Worker
 *
 * This script is executed by a cron job in the background. It checks for
 * pending AI learning jobs, processes them in a controlled manner, and updates
 * their status in the database. This prevents server timeouts for large folders.
 */

// Set a long execution time, as this is a background process.
// Note: This might be limited by server configuration.
set_time_limit(1800); // 30 minutes

// The script is inside the 'cron' folder, so we go up one level to the project root.
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/../app/helpers.php';

// --- MAIN CRON JOB LOGIC ---

try {
    $pdo = get_db_connection();

    // 1. Find a pending job to process.
    // We select one job that is 'pending' and lock it by setting its status to 'processing'.
    // The FOR UPDATE clause prevents other cron processes from picking up the same job.
    $pdo->beginTransaction();
    $stmt = $pdo->prepare("SELECT * FROM ai_learning_jobs WHERE status = 'pending' ORDER BY created_at ASC LIMIT 1 FOR UPDATE");
    $stmt->execute();
    $job = $stmt->fetch();

    if (!$job) {
        $pdo->commit();
        // echo "No pending jobs found.\n"; // Optional: for command-line debugging
        exit(); // No jobs to do, exit quietly.
    }

    // Lock the job by updating its status
    $update_stmt = $pdo->prepare("UPDATE ai_learning_jobs SET status = 'processing', started_at = NOW() WHERE id = ?");
    $update_stmt->execute([$job['id']]);
    $pdo->commit();

    // 2. Initialize Google Drive Service
    // -----------------------------------------------------------------------------
    $client = new Google_Client();
    $client->setAuthConfig(__DIR__ . '/../credentials.json');
    $client->setRedirectUri(GOOGLE_REDIRECT_URI);
    $client->setAccessType('offline');
    $client->addScope(Google_Service_Drive::DRIVE_READONLY);

    $tokenPath = __DIR__ . '/../token.json';
    if (!file_exists($tokenPath)) {
        throw new Exception("token.json not found. Please authenticate via the admin panel first.");
    }
    $accessToken = json_decode(file_get_contents($tokenPath), true);
    $client->setAccessToken($accessToken);

    if ($client->isAccessTokenExpired()) {
        if ($client->getRefreshToken()) {
            $client->fetchAccessTokenWithRefreshToken($client->getRefreshToken());
            file_put_contents($tokenPath, json_encode($client->getAccessToken()));
        } else {
            throw new Exception("Token expired and no refresh token available.");
        }
    }
    $drive_service = new Google_Service_Drive($client);

    // 3. Define the Recursive Processing Function
    // -----------------------------------------------------------------------------
    function processFilesInFolder($folder_id, $drive_service, $pdo, $job_id, &$processed_in_this_run, &$total_items_scanned) {
        // (This recursive logic remains the same as the previous version)
        $check_stmt = $pdo->prepare("SELECT 1 FROM processed_files WHERE file_id = ?");
        $exportable_mime_types = ['application/vnd.google-apps.document', 'application/vnd.google-apps.spreadsheet', 'application/vnd.google-apps.presentation'];
        
        $pageToken = null;
        do {
            $response = $drive_service->files->listFiles([
                'q' => "'{$folder_id}' in parents and trashed=false",
                'fields' => 'nextPageToken, files(id, name, mimeType)',
                'pageToken' => $pageToken
            ]);

            foreach ($response->getFiles() as $item) {
                $total_items_scanned++;
                if ($item->getMimeType() === 'application/vnd.google-apps.folder') {
                    processFilesInFolder($item->getId(), $drive_service, $pdo, $job_id, $processed_in_this_run, $total_items_scanned);
                } else {
                    $check_stmt->execute([$item->getId()]);
                    if (!$check_stmt->fetch()) {
                        if (in_array($item->getMimeType(), $exportable_mime_types)) {
                            $content = $drive_service->files->export($item->getId(), 'text/plain', ['alt' => 'media']);
                        } else {
                            $content = $drive_service->files->get($item->getId(), ['alt' => 'media']);
                        }
                        $file_content = $content->getBody()->getContents();
                        $text_content = $file_content;
                        $embedding_vector = "SIMULATED_VECTOR_FOR_" . substr(md5($text_content), 0, 10);

                        $pdo->beginTransaction();
                        $processed_stmt = $pdo->prepare("INSERT INTO processed_files (file_id, file_name) VALUES (?, ?)");
                        $processed_stmt->execute([$item->getId(), $item->getName()]);
                        $embedding_stmt = $pdo->prepare("INSERT INTO vector_embeddings (file_id, content_chunk, embedding) VALUES (?, ?, ?)");
                        $embedding_stmt->execute([$item->getId(), mb_substr($text_content, 0, 500), $embedding_vector]);
                        $pdo->commit();
                        $processed_in_this_run++;
                    }
                }
                 // Update progress in the database
                $progress_stmt = $pdo->prepare("UPDATE ai_learning_jobs SET processed_items = ?, total_items = ? WHERE id = ?");
                $progress_stmt->execute([$processed_in_this_run, $total_items_scanned, $job_id]);
            }
            $pageToken = $response->getNextPageToken();
        } while ($pageToken != null);
    }

    // 4. Execute the Learning Process
    // -----------------------------------------------------------------------------
    $initial_folder_id = ($job['job_type'] === 'cv') ? GOOGLE_DRIVE_CV_FOLDER_ID : GOOGLE_DRIVE_JOB_POSTER_FOLDER_ID;
    $processed_count = 0;
    $total_scanned = 0;
    
    processFilesInFolder($initial_folder_id, $drive_service, $pdo, $job['id'], $processed_count, $total_scanned);

    // 5. Finalize the Job
    // -----------------------------------------------------------------------------
    $final_message = "Scan complete. Learned from {$processed_count} new files out of {$total_scanned} total items scanned.";
    $finalize_stmt = $pdo->prepare("UPDATE ai_learning_jobs SET status = 'completed', completed_at = NOW(), message = ?, processed_items = ?, total_items = ? WHERE id = ?");
    $finalize_stmt->execute([$final_message, $processed_count, $total_scanned, $job['id']]);

} catch (Exception $e) {
    // If an error occurs, mark the job as 'failed' and log the error message.
    if (isset($job['id']) && isset($pdo)) {
        $fail_stmt = $pdo->prepare("UPDATE ai_learning_jobs SET status = 'failed', completed_at = NOW(), message = ? WHERE id = ?");
        $fail_stmt->execute([$e->getMessage(), $job['id']]);
        if($pdo->inTransaction()) { $pdo->rollBack(); }
    }
    // You should also log this error to a file for debugging.
    // error_log("Cron job failed: " . $e->getMessage());
}

