<?php
/**
 * AI Feedback Loop Learner Cron Job
 *
 * This script runs periodically in the background to make the AI "smarter".
 * It finds completed interview data (CV, job ad, transcripts), simulates
 * converting that text into vector embeddings, and adds them to the AI's
 * knowledge base (`vector_embeddings` table).
 */

// Set a long execution time suitable for a background task.
set_time_limit(1800); // 30 minutes

// Use __DIR__ to ensure paths are correct when run from any location by the cron daemon.
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/../app/helpers.php';

// --- For command-line output ---
echo "Starting feedback loop learning process at " . date('Y-m-d H:i:s') . "\n";

try {
    $pdo = get_db_connection();

    // Find recently completed user interviews that haven't been processed for learning yet.
    // NOTE: A production system would add a column like `learned_by_ai` (tinyint)
    // to the `interview_reports` table to track this and query for `learned_by_ai = 0`.
    // For this simulation, we'll just re-process reports from the last 24 hours.
    $stmt = $pdo->query("
        SELECT ir.user_id, ir.id as report_id
        FROM interview_reports ir
        WHERE ir.generated_at > DATE_SUB(NOW(), INTERVAL 1 DAY)
    ");
    $reports = $stmt->fetchAll();

    if (empty($reports)) {
        echo "No new reports found to learn from.\n";
        exit();
    }

    echo "Found " . count($reports) . " new report(s) to process.\n";

    $total_chunks_learned = 0;

    foreach ($reports as $report) {
        $user_id = $report['user_id'];
        echo "Processing report ID #{$report['report_id']} for user ID #{$user_id}...\n";
        
        // 1. Fetch all text associated with this user's last session
        $learning_chunks = [];

        // a) Get Transcripts
        $stmt_transcripts = $pdo->prepare("SELECT question_text, answer_transcript FROM interview_results WHERE user_id = ?");
        $stmt_transcripts->execute([$user_id]);
        $transcripts = $stmt_transcripts->fetchAll();
        foreach ($transcripts as $t) {
            $learning_chunks[] = "Question: " . $t['question_text'] . " Answer: " . $t['answer_transcript'];
        }

        // b) Get content from Uploaded file paths (simulation)
        $stmt_uploads = $pdo->prepare("SELECT stored_filepath FROM user_uploads WHERE user_id = ?");
        $stmt_uploads->execute([$user_id]);
        $upload_paths = $stmt_uploads->fetchAll(PDO::FETCH_COLUMN);
        foreach ($upload_paths as $path) {
            // In a real application, you would use OCR/Tika/PyPDF2 libraries here
            // to extract actual text content from the files stored at $path.
            $learning_chunks[] = "SIMULATED content from file: " . basename($path);
        }

        // 2. Process and vectorize this text (Simulation)
        if (empty($learning_chunks)) {
            echo "  - No text chunks found for this report. Skipping.\n";
            continue;
        }

        // 3. Add to vector_embeddings table
        $stmt_insert = $pdo->prepare("INSERT INTO vector_embeddings (file_id, content_chunk, embedding) VALUES (?, ?, ?)");
        
        foreach ($learning_chunks as $chunk) {
            $file_id = 'user_session_' . $user_id . '_report_' . $report['report_id'];
            // In a real app, this is where you'd call an embedding API (e.g., OpenAI, Vertex AI)
            $embedding_vector = "SIMULATED_FEEDBACK_VECTOR_" . substr(md5($chunk), 0, 10);
            
            $stmt_insert->execute([$file_id, mb_substr($chunk, 0, 500), $embedding_vector]);
            $total_chunks_learned++;
            echo "  - Learned from chunk: '" . mb_substr($chunk, 0, 50) . "...'\n";
        }
        
        // 4. Mark this report as learned (In a real app, this would be uncommented)
        // $stmt_mark = $pdo->prepare("UPDATE interview_reports SET learned_by_ai = 1 WHERE id = ?");
        // $stmt_mark->execute([$report['report_id']]);
        echo "  - Finished processing report ID #{$report['report_id']}.\n";
    }

    echo "--------------------------------------------------\n";
    echo "Feedback loop learning process completed successfully.\n";
    echo "Total new text chunks learned: {$total_chunks_learned}\n";

} catch (Exception $e) {
    echo "An error occurred: " . $e->getMessage() . "\n";
    error_log("Feedback Loop Cron failed: " . $e->getMessage());
}

