<?php
/**
 * Scorer Logika Tes Psikologi DISC
 *
 * File ini berisi logika inti (yang berasal dari formula Excel Anda)
 * untuk mengubah jawaban mentah tes DISC dari pengguna menjadi
 * laporan skor yang terstruktur.
 *
 * File ini dipanggil oleh: v1/actions/submit_psychotest_handler.php
 */

/**
 * Menghitung skor DISC (Most, Least, Change) dari array jawaban.
 *
 * @param array $answers Array jawaban yang diterima dari $_POST.
 * Format yang diharapkan:
 * [
 * 'disc_q1' => ['most' => 'A', 'least' => 'B'],
 * 'disc_q2' => ['most' => 'D', 'least' => 'A'],
 * ...
 * ]
 *
 * @return array Array terstruktur yang berisi semua skor
 * (Most, Least, Change) dan profil dominan.
 */
function calculateDiscReport(array $answers): array
{
    // =======================================================================
    // KUNCI JAWABAN (WAJIB DISESUAIKAN DENGAN FILE EXCEL ANDA)
    // =======================================================================

    /**
     * Peta untuk Skor "Most" (Graph 1 - Public Self)
     * Format: [Nomor Pertanyaan][Pilihan Jawaban] => 'Dimensi'
     * '*' (bintang) sering digunakan dalam DISC untuk jawaban "tidak ada",
     * yang tidak dihitung dalam D, I, S, atau C.
     */
    $discMostMap = [
        // Q#  [ A => ?,  B => ?,  C => ?,  D => ? ]
        1 => ['A' => 'D', 'B' => 'I', 'C' => 'S', 'D' => 'C'],
        2 => ['A' => 'I', 'B' => 'D', 'C' => 'C', 'D' => 'S'],
        3 => ['A' => 'S', 'B' => 'I', 'C' => 'D', 'D' => 'C'],
        4 => ['A' => 'C', 'B' => 'S', 'C' => 'I', 'D' => 'D'],
        5 => ['A' => '*', 'B' => 'C', 'C' => 'D', 'D' => 'I'], // Contoh dengan '*'
        6 => ['A' => 'D', 'B' => 'S', 'C' => 'C', 'D' => 'I'],
        7 => ['A' => 'I', 'B' => 'C', 'C' => 'S', 'D' => 'D'],
        8 => ['A' => 'S', 'B' => 'D', 'C' => 'I', 'D' => 'C'],
        9 => ['A' => 'C', 'B' => 'I', 'C' => 'D', 'D' => 'S'],
        10 => ['A' => 'D', 'B' => 'S', 'C' => 'I', 'D' => 'C'],
        11 => ['A' => 'I', 'B' => 'D', 'C' => 'S', 'D' => 'C'],
        12 => ['A' => 'S', 'B' => 'C', 'C' => 'D', 'D' => 'I'],
        13 => ['A' => 'C', 'B' => 'S', 'C' => 'I', 'D' => 'D'],
        14 => ['A' => 'D', 'B' => 'I', 'C' => 'C', 'D' => 'S'],
        15 => ['A' => 'I', 'B' => 'D', 'C' => 'S', 'D' => 'C'],
        16 => ['A' => 'S', 'B' => 'C', 'C' => 'D', 'D' => 'I'],
        17 => ['A' => 'C', 'B' => 'S', 'C' => 'I', 'D' => 'D'],
        18 => ['A' => 'D', 'B' => 'I', 'C' => 'S', 'D' => 'C'],
        19 => ['A' => 'I', 'B' => 'D', 'C' => 'C', 'D' => 'S'],
        20 => ['A' => 'S', 'B' => 'C', 'C' => 'D', 'D' => 'I'],
        21 => ['A' => 'C', 'B' => 'S', 'C' => 'I', 'D' => 'D'],
        22 => ['A' => 'D', 'B' => 'I', 'C' => 'S', 'D' => 'C'],
        23 => ['A' => 'I', 'B' => 'D', 'C' => 'C', 'D' => 'S'],
        24 => ['A' => 'S', 'B' => 'C', 'C' => 'D', 'D' => 'I'],
    ];

    /**
     * Peta untuk Skor "Least" (Graph 2 - Private Self)
     * Seringkali, peta ini sama dengan peta "Most".
     * Jika di Excel Anda berbeda, sesuaikan di sini.
     */
    $discLeastMap = $discMostMap; // Asumsi petanya sama

    // =======================================================================
    // PROSES PERHITUNGAN SKOR
    // =======================================================================

    // 1. Inisialisasi wadah skor
    $scores = [
        'most' => ['D' => 0, 'I' => 0, 'S' => 0, 'C' => 0, '*' => 0],
        'least' => ['D' => 0, 'I' => 0, 'S' => 0, 'C' => 0, '*' => 0],
        'change' => ['D' => 0, 'I' => 0, 'S' => 0, 'C' => 0],
    ];

    $total_questions = 24; // Sesuaikan jika Anda menggunakan 28 pertanyaan

    // 2. Loop melalui semua pertanyaan dan hitung skor "Most" dan "Least"
    for ($i = 1; $i <= $total_questions; $i++) {
        
        // Ambil jawaban 'most' dan 'least' untuk pertanyaan ke-$i
        $mostAnswer = $answers["disc_q{$i}"]['most'] ?? null;  // cth: 'A'
        $leastAnswer = $answers["disc_q{$i}"]['least'] ?? null; // cth: 'B'

        // Hitung Skor "Most" (Graph 1)
        if ($mostAnswer && isset($discMostMap[$i][$mostAnswer])) {
            $dimension = $discMostMap[$i][$mostAnswer]; // cth: 'D'
            $scores['most'][$dimension]++;
        }

        // Hitung Skor "Least" (Graph 2)
        if ($leastAnswer && isset($discLeastMap[$i][$leastAnswer])) {
            $dimension = $discLeastMap[$i][$leastAnswer]; // cth: 'I'
            $scores['least'][$dimension]++;
        }
    }

    // 3. Hitung Skor "Change" (Graph 3)
    // Rumus: Most - Least = Change
    $scores['change']['D'] = $scores['most']['D'] - $scores['least']['D'];
    $scores['change']['I'] = $scores['most']['I'] - $scores['least']['I'];
    $scores['change']['S'] = $scores['most']['S'] - $scores['least']['S'];
    $scores['change']['C'] = $scores['most']['C'] - $scores['least']['C'];

    // 4. Tentukan Profil Dominan (Penyederhanaan)
    // Ini adalah analisis sederhana. Analisis DISC nyata bisa jauh lebih kompleks
    // (melihat kombinasi, skor di atas/bawah garis tengah, dll).
    // Kita ambil skor 'Most' (Public Self) tertinggi sebagai profil dominan.
    
    // Hapus skor '*' sebelum mencari nilai tertinggi
    $public_scores = $scores['most'];
    unset($public_scores['*']);
    
    // Cari skor tertinggi
    $dominant_profile = array_keys($public_scores, max($public_scores));
    
    // Handle jika ada >1 skor tertinggi (cth: "DI" atau "IC")
    $dominant_profile_string = implode('', $dominant_profile);

    // =======================================================================
    // KEMBALIKAN HASIL
    // =======================================================================

    $result = [
        'scores' => [
            'graph1_most' => $scores['most'],
            'graph2_least' => $scores['least'],
            'graph3_change' => $scores['change'],
        ],
        'dominant_profile' => $dominant_profile_string,
        'summary' => [ // Anda bisa tambahkan interpretasi singkat di sini
            'D' => $scores['change']['D'], // Seringkali Graph 3 yang digunakan untuk interpretasi
            'I' => $scores['change']['I'],
            'S' => $scores['change']['S'],
            'C' => $scores['change']['C'],
        ]
    ];

    // Fungsi ini akan mengembalikan array PHP.
    // Handler (submit_psychotest_handler.php) akan
    // mengubahnya menjadi JSON untuk disimpan di database.
    return $result;
}