<?php
/**
 * Application Helper Functions
 */

// FIXED: Add the required 'use' statements for PHPMailer at the top of the file.
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;
use PHPMailer\PHPMailer\SMTP;

/**
 * Establishes a database connection using PDO.
 * Caches the connection to prevent multiple connections per request.
 * @return PDO The PDO database connection object.
 */
function get_db_connection() {
    static $pdo = null;
    if ($pdo === null) {
        try {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4";
            
            // Opsi ini sudah bagus, tapi kita akan mengaturnya secara eksplisit
            // SETELAH koneksi untuk debugging.
            $options = [
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES   => false,
            ];
            
            $pdo = new PDO($dsn, DB_USER, DB_PASS, $options);
            
            // --- PERBAIKAN DEBUGGING (TAMBAHKAN BARIS INI) ---
            // Ini adalah langkah debugging terpenting.
            // Ini memaksa PDO untuk melempar Exception jika kueri
            // INSERT, UPDATE, atau DELETE gagal.
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            
        } catch (PDOException $e) {
            // In a production environment, you should log this error instead of displaying it.
            error_log("Database connection failed: " . $e->getMessage());
            die("A database connection error occurred. Please try again later.");
        }
    }
    return $pdo;
}

/**
 * Checks if a regular user is logged in.
 * @return bool True if logged in, false otherwise.
 */
function is_logged_in() {
    return isset($_SESSION['user_id']);
}

/**
 * Checks if an administrator is logged in.
 * @return bool True if logged in, false otherwise.
 */
function is_admin() {
    return isset($_SESSION['admin_id']);
}

/**
 * Escapes a string for safe HTML output to prevent XSS attacks.
 * @param string|null $string The string to escape.
 * @return string The escaped string.
 */
function e($string) {
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

/**
 * Retrieves a translated string based on the current language.
 * @param string $key The translation key.
 * @return string The translated string or the key itself if not found.
 */
function trans($key) {
    static $translations = [];
    
    $lang = $_SESSION['lang'] ?? DEFAULT_LANG;

    if (!isset($translations[$lang])) {
        $lang_file = __DIR__ . '/../languages/' . $lang . '.php';
        if (file_exists($lang_file)) {
            $translations[$lang] = require $lang_file;
        } else {
            // Fallback to default language if the selected language file doesn't exist.
            $default_lang_file = __DIR__ . '/../languages/' . DEFAULT_LANG . '.php';
            $translations[$lang] = file_exists($default_lang_file) ? require $default_lang_file : [];
        }
    }

    return $translations[$lang][$key] ?? $key;
}

/**
 * Returns the full URL for a theme asset (CSS, JS, etc.).
 * @param string $path The path to the asset within the theme directory.
 * @return string The absolute URL to the asset.
 */
function asset_url($path) {
    return SITE_URL . '/public/themes/' . THEME . '/' . ltrim($path, '/');
}


/**
 * Sends an email using PHPMailer with SMTP configuration.
 * @param string $to The recipient's email address.
 * @param string $subject The email subject.
 * @param string $body The HTML email body.
 * @return bool True on success, false on failure.
 */
function send_email($to, $subject, $body) {
    $mail = new PHPMailer(true);
    try {
        // Server settings
        // $mail->SMTPDebug = SMTP::DEBUG_SERVER; // Enable for detailed debug output
        $mail->isSMTP();
        $mail->Host       = SMTP_HOST;
        $mail->SMTPAuth   = true;
        $mail->Username   = SMTP_USER;
        $mail->Password   = SMTP_PASS;
        $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS; // Use 'tls' if port is 587
        $mail->Port       = SMTP_PORT;

        // Recipients
        $mail->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
        $mail->addAddress($to);

        // Content
        $mail->isHTML(true);
        $mail->Subject = $subject;
        $mail->Body    = $body;
        $mail->AltBody = strip_tags($body); // Plain text version for non-HTML mail clients

        $mail->send();
        return true;
    } catch (Exception $e) {
        // In a production environment, log the detailed error.
        error_log("Mailer Error: {$mail->ErrorInfo}");
        return false;
    }
}