<?php
/**
 * User Registration Handler
 *
 * This script processes the user registration form submission.
 * It validates input, checks for existing users, creates the new user account,
 * and sends a verification email.
 */

// Start the session to manage user feedback (messages).
session_start();

// Include core files
require_once '../config.php';
// FIXED: Add the Composer autoloader to make PHPMailer and other libraries available.
require_once '../vendor/autoload.php';
require_once '../app/helpers.php'; // Contains send_email() and get_db_connection()

// 1. VERIFY REQUEST METHOD
// -----------------------------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: ../index.php?page=register');
    exit();
}

// 2. RETRIEVE & SANITIZE FORM DATA
// -----------------------------------------------------------------------------
$full_name = trim($_POST['full_name'] ?? '');
$email = trim($_POST['email'] ?? '');
$password = $_POST['password'] ?? '';
$confirm_password = $_POST['confirm_password'] ?? '';
$errors = [];

// 3. VALIDATE INPUT
// -----------------------------------------------------------------------------
if (empty($full_name)) {
    $errors[] = "Full name is required.";
}
if (empty($email)) {
    $errors[] = "Email is required.";
} elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    $errors[] = "Invalid email format.";
}
if (empty($password)) {
    $errors[] = "Password is required.";
} elseif (strlen($password) < 8) {
    $errors[] = "Password must be at least 8 characters long.";
}
if ($password !== $confirm_password) {
    $errors[] = "Passwords do not match.";
}

// If there are validation errors, redirect back to the registration page.
if (!empty($errors)) {
    $_SESSION['error_messages'] = $errors;
    $_SESSION['form_data'] = ['full_name' => $full_name, 'email' => $email]; // Preserve form data
    header('Location: ../index.php?page=register');
    exit();
}

// 4. DATABASE OPERATIONS
// -----------------------------------------------------------------------------
try {
    $pdo = get_db_connection();

    // Check if the email already exists to prevent duplicates.
    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
    $stmt->execute([$email]);
    if ($stmt->fetch()) {
        $_SESSION['error_messages'] = ["An account with this email address already exists."];
        $_SESSION['form_data'] = ['full_name' => $full_name, 'email' => $email];
        header('Location: ../index.php?page=register');
        exit();
    }

    // Hash the password for secure storage.
    $hashed_password = password_hash($password, PASSWORD_DEFAULT);

    // Generate a unique token for email verification.
    $verification_token = bin2hex(random_bytes(32));

    // Insert the new user into the database.
    $stmt = $pdo->prepare(
        "INSERT INTO users (full_name, email, password, verification_token) VALUES (?, ?, ?, ?)"
    );
    $stmt->execute([$full_name, $email, $hashed_password, $verification_token]);

    // 5. SEND VERIFICATION EMAIL
    // -----------------------------------------------------------------------------
    $verification_link = SITE_URL . "/verify_email.php?token=" . $verification_token;
    $email_subject = "Verify Your Email Address for " . SITE_NAME;
    $email_body = "
        <h2>Thank you for registering!</h2>
        <p>Please click the link below to verify your email address and complete your registration:</p>
        <p><a href='{$verification_link}'>{$verification_link}</a></p>
        <p>If you did not create an account, please ignore this email.</p>
    ";

    if (send_email($email, $email_subject, $email_body)) {
        // If email is sent successfully, show a success message.
        $_SESSION['success_message'] = "Registration successful! Please check your email to verify your account.";
        header('Location: ../index.php?page=login');
        exit();
    } else {
        // If email fails, show a generic error. In a real app, log the detailed error.
        $_SESSION['error_messages'] = ["Could not send verification email. Please contact support."];
        header('Location: ../index.php?page=register');
        exit();
    }

} catch (PDOException $e) {
    // In production, log the error instead of displaying it.
    // error_log("Registration failed: " . $e->getMessage());
    $_SESSION['error_messages'] = ["An unexpected database error occurred. Please try again later."];
    header('Location: ../index.php?page=register');
    exit();
}

