<?php
/**
 * Test Management Handler
 *
 * This script handles all Create, Read, Update, Delete (CRUD) operations
 * for psychological tests, questions, and answers, initiated from the admin panel.
 */

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once '../config.php';
require_once '../app/helpers.php';

// 1. SECURITY CHECKS
// -----------------------------------------------------------------------------
if (!is_admin()) {
    $_SESSION['error_message'] = "Unauthorized access.";
    header('Location: ../admin/index.php?page=login');
    exit();
}
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: ../admin/index.php?page=tests');
    exit();
}

// 2. DETERMINE THE ACTION
// -----------------------------------------------------------------------------
$action = $_POST['action'] ?? '';

try {
    $pdo = get_db_connection();

    switch ($action) {
        // --- TEST ACTIONS ---
        case 'add_test':
            $test_name = trim($_POST['test_name'] ?? '');
            $description = trim($_POST['description'] ?? '');
            if (empty($test_name)) {
                throw new Exception("Test name is required.");
            }
            $stmt = $pdo->prepare("INSERT INTO psychological_tests (test_name, description, created_by_admin_id) VALUES (?, ?, ?)");
            $stmt->execute([$test_name, $description, $_SESSION['admin_id']]);
            $_SESSION['success_message'] = "New test '{$test_name}' has been successfully created.";
            header('Location: ../admin/index.php?page=tests');
            exit();

        case 'edit_test':
            $test_id = filter_input(INPUT_POST, 'test_id', FILTER_VALIDATE_INT);
            $test_name = trim($_POST['test_name'] ?? '');
            $description = trim($_POST['description'] ?? '');
            if (!$test_id || empty($test_name)) {
                throw new Exception("Invalid data provided for editing the test.");
            }
            $stmt = $pdo->prepare("UPDATE psychological_tests SET test_name = ?, description = ? WHERE id = ?");
            $stmt->execute([$test_name, $description, $test_id]);
            $_SESSION['success_message'] = "Test details have been successfully updated.";
            header('Location: ../admin/index.php?page=edit_test&id=' . $test_id);
            exit();

        case 'delete_test':
            $test_id = filter_input(INPUT_POST, 'test_id', FILTER_VALIDATE_INT);
            if (!$test_id) {
                throw new Exception("Invalid test ID for deletion.");
            }
            $stmt = $pdo->prepare("DELETE FROM psychological_tests WHERE id = ?");
            $stmt->execute([$test_id]);
            $_SESSION['success_message'] = "The test and all its associated questions have been deleted.";
            header('Location: ../admin/index.php?page=tests');
            exit();

        // --- QUESTION & ANSWER ACTIONS ---
        case 'add_question':
            $test_id = filter_input(INPUT_POST, 'test_id', FILTER_VALIDATE_INT);
            $question_text = trim($_POST['question_text'] ?? '');
            $answers = $_POST['answers'] ?? [];
            $scores = $_POST['scores'] ?? [];

            if (!$test_id || empty($question_text) || count($answers) < 2) {
                throw new Exception("A question and at least two answers are required.");
            }

            $pdo->beginTransaction();

            // Insert the question
            $stmt_q = $pdo->prepare("INSERT INTO test_questions (test_id, question_text) VALUES (?, ?)");
            $stmt_q->execute([$test_id, $question_text]);
            $question_id = $pdo->lastInsertId();

            // Insert the answers
            $stmt_a = $pdo->prepare("INSERT INTO test_answers (question_id, answer_text, score) VALUES (?, ?, ?)");
            foreach ($answers as $index => $answer_text) {
                $answer_text = trim($answer_text);
                $score = filter_var($scores[$index] ?? 0, FILTER_VALIDATE_INT);
                if (!empty($answer_text)) {
                    $stmt_a->execute([$question_id, $answer_text, $score]);
                }
            }
            
            // FIXED: Changed table name from 'questions' to 'test_questions'
            $stmt_count = $pdo->prepare("UPDATE psychological_tests SET question_count = (SELECT COUNT(*) FROM test_questions WHERE test_id = ?) WHERE id = ?");
            $stmt_count->execute([$test_id, $test_id]);

            $pdo->commit();
            $_SESSION['success_message'] = "New question has been successfully added.";
            header('Location: ../admin/index.php?page=manage_questions&test_id=' . $test_id);
            exit();
            
        case 'delete_question':
            $question_id = filter_input(INPUT_POST, 'question_id', FILTER_VALIDATE_INT);
            $test_id = filter_input(INPUT_POST, 'test_id', FILTER_VALIDATE_INT); // For redirecting back
            if (!$question_id || !$test_id) {
                throw new Exception("Invalid question ID for deletion.");
            }
            
            $pdo->beginTransaction();
            
            $stmt = $pdo->prepare("DELETE FROM questions WHERE id = ?");
            $stmt->execute([$question_id]);
            
            // FIXED: Changed table name from 'questions' to 'test_questions'
            $stmt_count = $pdo->prepare("UPDATE psychological_tests SET question_count = (SELECT COUNT(*) FROM test_questions WHERE test_id = ?) WHERE id = ?");
            $stmt_count->execute([$test_id, $test_id]);
            
            $pdo->commit();
            
            $_SESSION['success_message'] = "The question has been successfully deleted.";
            header('Location: ../admin/index.php?page=manage_questions&test_id=' . $test_id);
            exit();

        default:
            throw new Exception("Invalid action specified.");
    }
} catch (Exception $e) {
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    $_SESSION['error_message'] = "An error occurred: " . $e->getMessage();
    $redirect_page = 'tests';
    if (isset($_POST['test_id'])) {
        $redirect_page = 'manage_questions&test_id=' . $_POST['test_id'];
    }
    header('Location: ../admin/index.php?page=' . $redirect_page);
    exit();
}

