<?php
/**
 * Handler Submit Psikotes (Versi Cerdas)
 *
 * File ini menggantikan handler yang lama. File ini dapat memproses SEMUA
 * jenis tes (Simple, DISC, PAPI) dengan:
 * 1. Memeriksa tipe tes dari database.
 * 2. Memanggil file scorer yang sesuai ('disc_scorer.php' atau 'papi_scorer.php').
 * 3. Menyimpan hasil ke tabel database yang benar.
 */

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// 1. MEMUAT DEPENDENSI INTI
// -----------------------------------------------------------------------------
require_once '../config.php';
require_once '../app/helpers.php';
// Memuat file "otak" untuk tes baru
require_once '../app/scoring/disc_scorer.php';
require_once '../app/scoring/papi_scorer.php';

// 2. KEAMANAN & VALIDASI
// -----------------------------------------------------------------------------
if (!is_logged_in() || $_SERVER['REQUEST_METHOD'] !== 'POST') {
    $_SESSION['error_message'] = "Unauthorized access.";
    header('Location: ../index.php?page=login');
    exit();
}

// Ambil data POST
$user_id = $_SESSION['user_id'];
$test_id = filter_input(INPUT_POST, 'test_id', FILTER_VALIDATE_INT);
$answers = $_POST; // Kirim SEMUA data POST ke scorer

if (!$test_id) {
    $_SESSION['error_message'] = "Invalid test ID submitted.";
    header('Location: ../index.php?page=psy_test');
    exit();
}

// 3. PROSES & SIMPAN HASIL
// -----------------------------------------------------------------------------
try {
    $pdo = get_db_connection();

    // 3a. Identifikasi Tipe Tes (Langkah Kunci)
    // Kita perlu tahu tes macam apa yang baru saja diselesaikan
    $stmt_type = $pdo->prepare("SELECT test_type FROM psychological_tests WHERE id = ?");
    $stmt_type->execute([$test_id]);
    $test_type = $stmt_type->fetchColumn();

    if (!$test_type) {
        throw new Exception("Test with ID {$test_id} could not be found.");
    }

    // 3b. Hitung Skor/Laporan Berdasarkan Tipe Tes
    $simple_score_result = null;
    $complex_report_json = null;

    if ($test_type === 'disc') {
        $report_array = calculateDiscReport($answers);
        $complex_report_json = json_encode($report_array);
        
    } elseif ($test_type === 'papi') {
        $report_array = calculatePapiReport($answers);
        $complex_report_json = json_encode($report_array);
        
    } else {
        // Ini adalah logika 'simple' (Tipe Default / Lama)
        // Diambil dari psy_test_handler.php lama 
        $submitted_answers = $answers['answers'] ?? [];
        if (empty($submitted_answers)) {
            throw new Exception("No answers were submitted.");
        }
        
        $placeholders = implode(',', array_fill(0, count($submitted_answers), '?'));
        $stmt_scores = $pdo->prepare("SELECT score FROM test_answers WHERE id IN ($placeholders)");
        $stmt_scores->execute(array_values($submitted_answers));
        $scores = $stmt_scores->fetchAll(PDO::FETCH_COLUMN);
        
        // Hasil akhir untuk tes 'simple' hanyalah skor total
        $simple_score_result = array_sum($scores);
    }

    // 3c. Simpan Hasil ke Database (menggunakan Transaksi)
    $pdo->beginTransaction();

    if ($test_type === 'simple') {
        // Simpan ke tabel LAMA (user_test_results) 
        $stmt_result = $pdo->prepare(
            "INSERT INTO user_test_results (user_id, test_id, total_score) VALUES (?, ?, ?)"
        );
        $stmt_result->execute([$user_id, $test_id, $simple_score_result]);
        
    } else {
        // Simpan ke tabel BARU (user_psychotest_reports)
        $stmt_report = $pdo->prepare(
            "INSERT INTO user_psychotest_reports (user_id, test_id, report_data) VALUES (?, ?, ?)"
        );
        $stmt_report->execute([$user_id, $test_id, $complex_report_json]);
    }

    // 3d. Majukan Pengguna ke Langkah Berikutnya (Interview) 
    $stmt_step = $pdo->prepare("UPDATE users SET current_step = 4 WHERE id = ?");
    $stmt_step->execute([$user_id]);

    // Selesaikan transaksi
    $pdo->commit();

    // 4. REDIRECT KE LANGKAH BERIKUTNYA
    // -----------------------------------------------------------------------------
    header('Location: ../index.php?page=interview');
    exit();

} catch (Exception $e) {
    // 5. PENANGANAN ERROR
    // -----------------------------------------------------------------------------
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack(); // Batalkan transaksi jika terjadi error
    }
    
    // Kirim pengguna kembali ke halaman tes dengan pesan error
    // 
    $_SESSION['error_message'] = "An error occurred while submitting your test: " . $e->getMessage();
    header('Location: ../index.php?page=psy_test');
    exit();
}