<?php
/**
 * Midtrans Payment Notification Handler (Webhook) - Robust Version
 *
 * This script is the endpoint that Midtrans will send notifications to.
 * It now uses absolute paths to prevent errors when called by the Midtrans server.
 */

// Use absolute paths with __DIR__ for reliability in server-to-server calls.
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/../app/helpers.php';

// --- For logging purposes during development ---
// file_put_contents('notification_log.txt', file_get_contents('php://input') . "\n", FILE_APPEND);

// 1. CONFIGURE MIDTRANS
\Midtrans\Config::$serverKey = MIDTRANS_SERVER_KEY;
\Midtrans\Config::$isProduction = MIDTRANS_IS_PRODUCTION;

// 2. RECEIVE AND VALIDATE THE NOTIFICATION
$notif = null;
try {
    $notif = new \Midtrans\Notification();
} catch (Exception $e) {
    http_response_code(400);
    error_log('Midtrans Notification Error: ' . $e->getMessage());
    exit();
}

// 3. EXTRACT KEY INFORMATION
$order_id = $notif->order_id;
$transaction_status = $notif->transaction_status;
$fraud_status = $notif->fraud_status;
$payment_type = $notif->payment_type;
$gross_amount = $notif->gross_amount;
$signature_key = hash('sha512', $order_id . $notif->status_code . $gross_amount . MIDTRANS_SERVER_KEY);

// 4. SECURITY CHECK: VALIDATE THE SIGNATURE
if ($signature_key != $notif->signature_key) {
    http_response_code(403);
    error_log("Midtrans signature validation failed for order ID: {$order_id}");
    exit();
}

// 5. PROCESS THE NOTIFICATION BASED ON TRANSACTION STATUS
try {
    $pdo = get_db_connection();
    
    $stmt_find = $pdo->prepare("SELECT * FROM transactions WHERE order_id = ?");
    $stmt_find->execute([$order_id]);
    $transaction = $stmt_find->fetch();

    if (!$transaction) {
        http_response_code(404);
        error_log("Transaction not found for order ID: {$order_id}");
        exit();
    }

    // Process only if the transaction is still pending to prevent duplicate processing
    if ($transaction['status'] === 'pending') {
        if ($transaction_status == 'capture' || $transaction_status == 'settlement') {
            if ($fraud_status == 'accept') {
                // PAYMENT SUCCESSFUL AND SECURE
                $stmt_update_trans = $pdo->prepare("UPDATE transactions SET status = 'paid', payment_method = ? WHERE order_id = ?");
                $stmt_update_trans->execute([$payment_type, $order_id]);

                // Advance the user to the next step
                $stmt_update_user = $pdo->prepare("UPDATE users SET current_step = 3 WHERE id = ?");
                $stmt_update_user->execute([$transaction['user_id']]);
            }
        } else if ($transaction_status == 'cancel' || $transaction_status == 'deny' || $transaction_status == 'expire') {
            // PAYMENT FAILED OR EXPIRED
            $stmt_update_trans = $pdo->prepare("UPDATE transactions SET status = 'failed' WHERE order_id = ?");
            $stmt_update_trans->execute([$order_id]);
        }
    }

    // 6. RESPOND TO MIDTRANS
    http_response_code(200);

} catch (Exception $e) {
    http_response_code(500);
    error_log("Database error while processing Midtrans notification for order ID {$order_id}: " . $e->getMessage());
    exit();
}

